/* Design Philosophy: Modern Industrial Minimalism
   - Clean stat cards with count-up animation
   - Teal gradient background
*/

import { Users, Target, CheckCircle, Award } from "lucide-react";
import { useEffect, useRef, useState } from "react";

interface StatItemProps {
  icon: React.ElementType;
  value: number;
  label: string;
  delay: number;
}

function StatItem({ icon: Icon, value, label, delay }: StatItemProps) {
  const [count, setCount] = useState(0);
  const [isVisible, setIsVisible] = useState(false);
  const ref = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true);
        }
      },
      { threshold: 0.3 }
    );

    if (ref.current) {
      observer.observe(ref.current);
    }

    return () => {
      if (ref.current) {
        observer.unobserve(ref.current);
      }
    };
  }, []);

  useEffect(() => {
    if (!isVisible) return;

    const duration = 2000;
    const steps = 60;
    const increment = value / steps;
    let current = 0;

    const timer = setInterval(() => {
      current += increment;
      if (current >= value) {
        setCount(value);
        clearInterval(timer);
      } else {
        setCount(Math.floor(current));
      }
    }, duration / steps);

    return () => clearInterval(timer);
  }, [isVisible, value]);

  return (
    <div 
      ref={ref}
      className="text-center animate-count-up opacity-0"
      style={{ animationDelay: `${delay}s`, animationFillMode: 'forwards' }}
    >
      <div className="inline-flex items-center justify-center w-16 h-16 bg-primary/20 rounded-sm mb-4">
        <Icon className="w-8 h-8 text-primary" />
      </div>
      <div className="text-5xl font-bold text-white mb-2" style={{ fontFamily: 'Rajdhani, sans-serif' }}>
        {count}+
      </div>
      <div className="text-white/80 uppercase tracking-wide text-sm" style={{ fontFamily: 'Bebas Neue, sans-serif' }}>
        {label}
      </div>
    </div>
  );
}

export default function Stats() {
  const stats = [
    { icon: Users, value: 454, label: "Mutlu Müşteri" },
    { icon: Target, value: 650, label: "Sorun Tespiti" },
    { icon: CheckCircle, value: 560, label: "Sorun Çözümü" },
    { icon: Award, value: 10, label: "Yıllık Tecrübe" },
  ];

  return (
    <section className="py-20 bg-gradient-to-br from-secondary via-primary to-secondary relative overflow-hidden">
      {/* Technical Pattern Overlay */}
      <div className="absolute inset-0 technical-pattern opacity-5" />
      
      <div className="container relative z-10">
        <div className="grid grid-cols-2 lg:grid-cols-4 gap-8 lg:gap-12">
          {stats.map((stat, index) => (
            <StatItem
              key={index}
              icon={stat.icon}
              value={stat.value}
              label={stat.label}
              delay={index * 0.1}
            />
          ))}
        </div>
      </div>
    </section>
  );
}
