/* Design Philosophy: Modern Industrial Minimalism
   - Clean geometric forms with teal accent
   - Asymmetric layout with left-weighted composition
   - Rajdhani for logo, Inter for navigation
*/

import { Phone, Mail, Facebook, Twitter, Instagram, Youtube, Menu } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useState } from "react";
import {
  Sheet,
  SheetContent,
  SheetTrigger,
} from "@/components/ui/sheet";

export default function Header() {
  const [isOpen, setIsOpen] = useState(false);

  const navItems = [
    { label: "Anasayfa", href: "#home" },
    { label: "Hakkımızda", href: "#about" },
    { label: "Hizmetlerimiz", href: "#services" },
    { label: "Hizmet Bölgelerimiz", href: "#regions" },
    { label: "İletişim", href: "#contact" },
  ];

  const scrollToSection = (href: string) => {
    const element = document.querySelector(href);
    if (element) {
      element.scrollIntoView({ behavior: "smooth" });
      setIsOpen(false);
    }
  };

  return (
    <header className="fixed top-0 left-0 right-0 z-50 bg-background/95 backdrop-blur-sm border-b border-border">
      {/* Top Bar - Contact Info */}
      <div className="bg-secondary text-secondary-foreground">
        <div className="container">
          <div className="flex flex-wrap items-center justify-between py-2 text-sm gap-2">
            <div className="flex items-center gap-4 flex-wrap">
              <a href="mailto:info@recber-tesisat.com" className="flex items-center gap-2 hover:text-primary transition-colors">
                <Mail className="w-4 h-4" />
                <span className="hidden sm:inline">info@recber-tesisat.com</span>
              </a>
              <a href="tel:+905324456921" className="flex items-center gap-2 hover:text-primary transition-colors">
                <Phone className="w-4 h-4" />
                <span>0532 445 69 21</span>
              </a>
            </div>
            <div className="flex items-center gap-3">
              <a href="#" className="hover:text-primary transition-colors" aria-label="Facebook">
                <Facebook className="w-4 h-4" />
              </a>
              <a href="#" className="hover:text-primary transition-colors" aria-label="Twitter">
                <Twitter className="w-4 h-4" />
              </a>
              <a href="#" className="hover:text-primary transition-colors" aria-label="Instagram">
                <Instagram className="w-4 h-4" />
              </a>
              <a href="#" className="hover:text-primary transition-colors" aria-label="YouTube">
                <Youtube className="w-4 h-4" />
              </a>
            </div>
          </div>
        </div>
      </div>

      {/* Main Navigation */}
      <div className="container">
        <div className="flex items-center justify-between py-4">
          {/* Logo */}
          <a href="#home" className="flex items-center gap-3" onClick={(e) => { e.preventDefault(); scrollToSection("#home"); }}>
            <div className="w-12 h-12 bg-primary rounded-sm flex items-center justify-center">
              <span className="text-primary-foreground font-bold text-2xl" style={{ fontFamily: 'Rajdhani, sans-serif' }}>R</span>
            </div>
            <div className="flex flex-col">
              <span className="font-bold text-xl leading-none" style={{ fontFamily: 'Rajdhani, sans-serif' }}>REÇBER TESİSAT</span>
              <span className="text-xs text-muted-foreground uppercase tracking-wide" style={{ fontFamily: 'Bebas Neue, sans-serif' }}>Profesyonel Hizmet</span>
            </div>
          </a>

          {/* Desktop Navigation */}
          <nav className="hidden lg:flex items-center gap-1">
            {navItems.map((item) => (
              <Button
                key={item.href}
                variant="ghost"
                className="text-foreground hover:text-primary hover:bg-primary/10"
                onClick={() => scrollToSection(item.href)}
              >
                {item.label}
              </Button>
            ))}
          </nav>

          {/* Call Button */}
          <div className="hidden md:flex items-center gap-3">
            <a href="tel:+902163121715">
              <Button className="bg-primary hover:bg-primary/90 text-primary-foreground">
                <Phone className="w-4 h-4 mr-2" />
                0 (216) 312 17 15
              </Button>
            </a>
          </div>

          {/* Mobile Menu */}
          <Sheet open={isOpen} onOpenChange={setIsOpen}>
            <SheetTrigger asChild className="lg:hidden">
              <Button variant="outline" size="icon">
                <Menu className="w-5 h-5" />
              </Button>
            </SheetTrigger>
            <SheetContent side="right" className="w-[280px]">
              <nav className="flex flex-col gap-4 mt-8">
                {navItems.map((item) => (
                  <Button
                    key={item.href}
                    variant="ghost"
                    className="justify-start text-lg"
                    onClick={() => scrollToSection(item.href)}
                  >
                    {item.label}
                  </Button>
                ))}
                <a href="tel:+902163121715" className="mt-4">
                  <Button className="w-full bg-primary hover:bg-primary/90 text-primary-foreground">
                    <Phone className="w-4 h-4 mr-2" />
                    0 (216) 312 17 15
                  </Button>
                </a>
              </nav>
            </SheetContent>
          </Sheet>
        </div>
      </div>
    </header>
  );
}
