/* Design Philosophy: Modern Industrial Minimalism
   - Contact form with company info
   - Clean layout with teal accents
*/

import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Phone, Mail, MapPin, Clock } from "lucide-react";
import { useState } from "react";
import { toast } from "sonner";

export default function Contact() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    message: "",
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    toast.success("Mesajınız alındı! En kısa sürede size dönüş yapacağız.");
    setFormData({ name: "", email: "", phone: "", message: "" });
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setFormData({ ...formData, [e.target.name]: e.target.value });
  };

  return (
    <section id="contact" className="py-20 bg-muted/30">
      <div className="container">
        {/* Section Header */}
        <div className="text-center max-w-3xl mx-auto mb-16">
          <span className="text-accent-caps text-primary text-sm">İletişim</span>
          <h2 className="text-4xl md:text-5xl font-bold mt-2 mb-4 text-foreground">
            Bize Ulaşın
          </h2>
          <p className="text-muted-foreground text-lg">
            Tesisat sorunlarınız için 7/24 hizmetinizdeyiz. Formu doldurun veya direkt arayın.
          </p>
        </div>

        <div className="grid lg:grid-cols-2 gap-12">
          {/* Contact Info */}
          <div className="space-y-8">
            <div>
              <h3 className="text-2xl font-bold mb-6 text-foreground">İletişim Bilgileri</h3>
              <div className="space-y-6">
                <div className="flex items-start gap-4">
                  <div className="w-12 h-12 bg-primary/20 rounded-sm flex items-center justify-center flex-shrink-0">
                    <Phone className="w-6 h-6 text-primary" />
                  </div>
                  <div>
                    <div className="font-semibold text-foreground mb-1">Telefon</div>
                    <a href="tel:+902163121715" className="text-muted-foreground hover:text-primary">
                      0 (216) 312 17 15
                    </a>
                    <br />
                    <a href="tel:+905324456921" className="text-muted-foreground hover:text-primary">
                      0532 445 69 21
                    </a>
                  </div>
                </div>

                <div className="flex items-start gap-4">
                  <div className="w-12 h-12 bg-primary/20 rounded-sm flex items-center justify-center flex-shrink-0">
                    <Mail className="w-6 h-6 text-primary" />
                  </div>
                  <div>
                    <div className="font-semibold text-foreground mb-1">E-posta</div>
                    <a href="mailto:info@recber-tesisat.com" className="text-muted-foreground hover:text-primary">
                      info@recber-tesisat.com
                    </a>
                  </div>
                </div>

                <div className="flex items-start gap-4">
                  <div className="w-12 h-12 bg-primary/20 rounded-sm flex items-center justify-center flex-shrink-0">
                    <MapPin className="w-6 h-6 text-primary" />
                  </div>
                  <div>
                    <div className="font-semibold text-foreground mb-1">Hizmet Bölgesi</div>
                    <p className="text-muted-foreground">
                      İstanbul - Anadolu Yakası<br />
                      Tüm İlçeler
                    </p>
                  </div>
                </div>

                <div className="flex items-start gap-4">
                  <div className="w-12 h-12 bg-primary/20 rounded-sm flex items-center justify-center flex-shrink-0">
                    <Clock className="w-6 h-6 text-primary" />
                  </div>
                  <div>
                    <div className="font-semibold text-foreground mb-1">Çalışma Saatleri</div>
                    <p className="text-muted-foreground">
                      7/24 Acil Servis<br />
                      Hafta İçi ve Hafta Sonu
                    </p>
                  </div>
                </div>
              </div>
            </div>

            {/* Emergency Banner */}
            <div className="bg-accent text-white p-6 rounded-sm">
              <h4 className="text-xl font-bold mb-2" style={{ fontFamily: 'Rajdhani, sans-serif' }}>
                Acil Durum mu?
              </h4>
              <p className="mb-4">
                Su kaçağı, tıkanıklık gibi acil durumlar için hemen arayın!
              </p>
              <a href="tel:+902163121715">
                <Button className="bg-white text-accent hover:bg-white/90 w-full">
                  <Phone className="w-4 h-4 mr-2" />
                  Acil Arama: 0 (216) 312 17 15
                </Button>
              </a>
            </div>
          </div>

          {/* Contact Form */}
          <div className="bg-card border-2 border-border rounded-sm p-8">
            <h3 className="text-2xl font-bold mb-6 text-foreground">Mesaj Gönderin</h3>
            <form onSubmit={handleSubmit} className="space-y-6">
              <div>
                <label htmlFor="name" className="block text-sm font-medium mb-2 text-foreground">
                  Adınız Soyadınız *
                </label>
                <Input
                  id="name"
                  name="name"
                  value={formData.name}
                  onChange={handleChange}
                  required
                  placeholder="Adınız ve soyadınız"
                  className="w-full"
                />
              </div>

              <div>
                <label htmlFor="email" className="block text-sm font-medium mb-2 text-foreground">
                  E-posta Adresiniz *
                </label>
                <Input
                  id="email"
                  name="email"
                  type="email"
                  value={formData.email}
                  onChange={handleChange}
                  required
                  placeholder="ornek@email.com"
                  className="w-full"
                />
              </div>

              <div>
                <label htmlFor="phone" className="block text-sm font-medium mb-2 text-foreground">
                  Telefon Numaranız *
                </label>
                <Input
                  id="phone"
                  name="phone"
                  type="tel"
                  value={formData.phone}
                  onChange={handleChange}
                  required
                  placeholder="0532 XXX XX XX"
                  className="w-full"
                />
              </div>

              <div>
                <label htmlFor="message" className="block text-sm font-medium mb-2 text-foreground">
                  Mesajınız *
                </label>
                <Textarea
                  id="message"
                  name="message"
                  value={formData.message}
                  onChange={handleChange}
                  required
                  placeholder="Tesisat sorununuzu detaylı olarak açıklayın..."
                  rows={5}
                  className="w-full"
                />
              </div>

              <Button type="submit" className="w-full bg-primary hover:bg-primary/90 text-primary-foreground" size="lg">
                Mesaj Gönder
              </Button>
            </form>
          </div>
        </div>
      </div>
    </section>
  );
}
